<?php

namespace App\Http\Controllers\Auth;

use App\Http\Controllers\Controller;
use App\Models\User;
use Illuminate\Http\Request;
use Illuminate\Support\Facades\Auth;
use Illuminate\Support\Facades\Hash;
use Illuminate\Support\Facades\DB;
use Illuminate\Validation\ValidationException;

class AuthController extends Controller
{
    /**
     * Login user
     */
    public function login(Request $request)
    {
        $request->validate([
            'email' => 'required|email',
            'password' => 'required|string',
            'remember' => 'boolean',
        ]);

        $user = User::where('email', $request->email)->first();

        if (!$user || !Hash::check($request->password, $user->password)) {
            throw ValidationException::withMessages([
                'email' => ['The provided credentials are incorrect.'],
            ]);
        }

        if (!$user->is_active) {
            throw ValidationException::withMessages([
                'email' => ['Your account has been deactivated. Please contact support.'],
            ]);
        }

        // Create token for API authentication
        $token = $user->createToken('auth-token')->plainTextToken;

        // Load relationships
        $user->load(['userSubscriptions' => function($query) {
            $query->where('status', 'active')->latest();
        }, 'libraries']);

        $libraryId = $user->libraries->first()?->id;

        return response()->json([
            'user' => [
                'id' => $user->id,
                'name' => $user->name,
                'email' => $user->email,
                'phone' => $user->phone,
                'crn' => $user->crn,
                'role' => $user->role,
                'ca_level' => $user->ca_level,
                'is_active' => $user->is_active,
                'trial_used' => $user->trial_used,
                'trial_started_at' => $user->trial_started_at,
                'trial_ends_at' => $user->trial_ends_at,
                'created_at' => $user->created_at,
                'isApproved' => $user->is_active, // For frontend compatibility
                'role' => $user->role, // For frontend compatibility
                'library_id' => $libraryId,
                'active_subscription' => $user->activeSubscription()->with('subscriptionPlan')->first(),
                'profile_picture' => $user->profile_picture,
            ],
            'token' => $token,
            'message' => 'Login successful'
        ]);
    }

    /**
     * Register new user (Student)
     */
    public function register(Request $request)
    {
        $request->validate([
            'name' => 'required|string|max:255',
            'email' => 'required|email|unique:users,email',
            'phone' => 'required|string|regex:/^03\d{9}$/',
            'crn' => 'required|string|regex:/^CRN\d{6}$/|unique:users,crn',
            'ca_level' => 'required|in:PRC,CAP,Final',
            'password' => 'required|string|min:8',
            'password_confirmation' => 'required|same:password',
            'plan_id' => 'nullable|exists:subscription_plans,id',
        ]);




        // Determine trial end date based on plan or default
        $trialEndsAt = now()->addDays(7);
        $plan = null;

        if ($request->plan_id) {
            $plan = \App\Models\SubscriptionPlan::find($request->plan_id);
            if ($plan && $plan->free_trial_days > 0) {
                $trialEndsAt = now()->addDays($plan->free_trial_days);
            }
        }

        // Create user
        $user = User::create([
            'name' => $request->name,
            'email' => $request->email,
            'phone' => $request->phone,
            'crn' => $request->crn,
            'role' => 'student',
            'ca_level' => $request->ca_level,
            'password' => Hash::make($request->password),
            'is_active' => true,
            'trial_used' => true,
            'trial_started_at' => now(),
            'trial_ends_at' => $trialEndsAt,
        ]);

        // Create subscription if plan selected
        if ($plan) {
            $status = 'pending';
            $expiresAt = null;
            $startedAt = null;

            if ($plan->free_trial_days > 0) {
                $status = 'active';
                $startedAt = now();
                $expiresAt = now()->addDays($plan->free_trial_days);
            } elseif ($plan->price == 0) {
                $status = 'active';
                $startedAt = now();
                $expiresAt = now()->addDays($plan->duration_days);
            }

            \App\Models\UserSubscription::create([
                'user_id' => $user->id,
                'subscription_plan_id' => $plan->id,
                'status' => $status,
                'started_at' => $startedAt,
                'expires_at' => $expiresAt,
                'auto_renew' => false,
                'amount_paid' => 0,
                'renewal_attempts' => 0,
            ]);
        }



        // Create token
        $token = $user->createToken('auth-token')->plainTextToken;

        return response()->json([
            'user' => [
                'id' => $user->id,
                'name' => $user->name,
                'email' => $user->email,
                'phone' => $user->phone,
                'crn' => $user->crn,
                'role' => $user->role,
                'ca_level' => $user->ca_level,
                'is_active' => $user->is_active,
                'trial_used' => $user->trial_used,
                'trial_started_at' => $user->trial_started_at,
                'trial_ends_at' => $user->trial_ends_at,
                'isApproved' => true,
                'role' => 'student',
                'library_id' => null,
                'active_subscription' => $user->activeSubscription()->with('subscriptionPlan')->first(),
                'profile_picture' => $user->profile_picture,
            ],
            'token' => $token,
            'message' => 'Registration successful! Welcome to SMART LIB.'
        ], 201);
    }

    /**
     * Send OTP for registration
     */
    public function sendOTP(Request $request)
    {
        $request->validate([
            'phone' => 'required|string|regex:/^03\d{9}$/',
            'crn' => 'required|string|regex:/^CRN\d{6}$/',
        ]);


        // Check if user already exists
        if (User::where('crn', $request->crn)->exists()) {
            throw ValidationException::withMessages([
                'crn' => ['This CRN is already registered.'],
            ]);
        }

        if (User::where('phone', $request->phone)->exists()) {
            throw ValidationException::withMessages([
                'phone' => ['This phone number is already registered.'],
            ]);
        }

        // Generate 4-digit OTP
        $otp = str_pad(rand(0, 9999), 4, '0', STR_PAD_LEFT);

        // Store OTP in database
        DB::table('otp_verifications')->insert([
            'phone' => $request->phone,
            'crn' => $request->crn,
            'otp' => $otp,
            'verified' => false,
            'expires_at' => now()->addMinutes(10),
            'attempts' => 0,
            'created_at' => now(),
            'updated_at' => now(),
        ]);

        // In production, send OTP via SMS
        // For demo, return OTP in response (REMOVE IN PRODUCTION!)
        return response()->json([
            'message' => 'OTP sent successfully to ' . $request->phone,
            'otp' => $otp, // REMOVE IN PRODUCTION!
            'expires_in' => 600, // 10 minutes
        ]);
    }

    /**
     * Get current authenticated user
     */
    public function me(Request $request)
    {
        $user = $request->user();
        
        $user->load(['userSubscriptions' => function($query) {
            $query->where('status', 'active')->latest();
        }, 'libraries']);

        $libraryId = $user->libraries->first()?->id;

        return response()->json([
            'id' => $user->id,
            'name' => $user->name,
            'email' => $user->email,
            'phone' => $user->phone,
            'crn' => $user->crn,
            'role' => $user->role,
            'ca_level' => $user->ca_level,
            'is_active' => $user->is_active,
            'trial_used' => $user->trial_used,
            'trial_started_at' => $user->trial_started_at,
            'trial_ends_at' => $user->trial_ends_at,
            'created_at' => $user->created_at,
            'isApproved' => $user->is_active,
            'role' => $user->role,
            'library_id' => $libraryId,
            'active_subscription' => $user->activeSubscription()->with('subscriptionPlan')->first(),
            'profile_picture' => $user->profile_picture,
        ]);
    }

    /**
     * Logout user
     */
    public function logout(Request $request)
    {
        // Revoke current token
        $request->user()->currentAccessToken()->delete();

        return response()->json([
            'message' => 'Logged out successfully'
        ]);
    }

    /**
     * Logout from all devices
     */
    public function logoutAll(Request $request)
    {
        // Revoke all tokens
        $request->user()->tokens()->delete();

        return response()->json([
            'message' => 'Logged out from all devices successfully'
        ]);
    }

    /**
     * Check field uniqueness
     */
    public function checkUniqueness(Request $request)
    {
        $request->validate([
            'field' => 'required|string|in:email,crn',
            'value' => 'required|string',
        ]);

        $field = $request->field;
        $value = trim($request->value);

        if ($field === 'email') {
            $exists = User::withTrashed()->where('email', $value)->exists();
        } else {
            $exists = User::withTrashed()->where($field, $value)->exists();
        }
        
        return response()->json([
            'exists' => $exists,
            'message' => $exists ? "This " . strtoupper($field === 'crn' ? 'CRN' : $field) . " is already registered." : null,
            'debug' => [
                'field' => $field,
                'value' => $value,
                'count' => User::where($field, $value)->count()
            ]
        ]);
    }
}
