<?php

namespace App\Http\Controllers\Student;

use App\Http\Controllers\Controller;
use App\Models\SeatBooking;
use App\Models\BookReservation;
use App\Models\Event;
use Illuminate\Http\Request;

class DashboardController extends Controller
{
    public function index(Request $request)
    {
        $user = $request->user();

        $stats = [
            'active_bookings' => SeatBooking::where('user_id', $user->id)
                ->whereIn('status', ['pending', 'active'])
                ->count(),
            'total_bookings' => SeatBooking::where('user_id', $user->id)->count(),
            'active_reservations' => BookReservation::where('user_id', $user->id)
                ->whereIn('status', ['pending', 'active'])
                ->count(),
            'upcoming_events' => Event::whereHas('registrations', function($query) use ($user) {
                $query->where('user_id', $user->id);
            })->where(function ($q) {
                $q->where('date', '>', now()->toDateString())
                  ->orWhere(function ($q2) {
                      $q2->where('date', '=', now()->toDateString())
                         ->where('start_time', '>', now()->toTimeString());
                  });
            })->count(),
            'loyalty_points' => $user->loyalty_points,
        ];

        $activeBooking = SeatBooking::with(['seat.library'])
            ->where('user_id', $user->id)
            ->whereIn('status', ['pending', 'active'])
            ->latest()
            ->first();

        $recentActivity = collect();

        // Get recent bookings
        $recentBookings = SeatBooking::with('seat.library')
            ->where('user_id', $user->id)
            ->latest()
            ->take(5)
            ->get()
            ->map(function($booking) {
                return [
                    'type' => 'booking',
                    'data' => $booking,
                    'created_at' => $booking->created_at,
                ];
            });

        // Get recent book reservations
        $recentReservations = BookReservation::with('book')
            ->where('user_id', $user->id)
            ->latest()
            ->take(5)
            ->get()
            ->map(function($reservation) {
                return [
                    'type' => 'reservation',
                    'data' => $reservation,
                    'created_at' => $reservation->created_at,
                ];
            });

        $recentActivity = $recentBookings->merge($recentReservations)
            ->sortByDesc('created_at')
            ->take(5)
            ->values();

        $upcomingEvents = Event::with('library')
            ->whereHas('registrations', function($query) use ($user) {
                $query->where('user_id', $user->id);
            })
            ->where(function ($q) {
                $q->where('date', '>', now()->toDateString())
                  ->orWhere(function ($q2) {
                      $q2->where('date', '=', now()->toDateString())
                         ->where('start_time', '>', now()->toTimeString());
                  });
            })
            ->orderBy('date')
            ->orderBy('start_time')
            ->take(3)
            ->get();

        // Calculate study analytics
        $completedBookings = SeatBooking::where('user_id', $user->id)
            ->where('status', 'checked_out')
            ->where('check_out_time', '>=', now()->subDays(7))
            ->get();

        $totalMinutes = $completedBookings->sum('total_minutes');
        $totalHoursToday = SeatBooking::where('user_id', $user->id)
            ->where('status', 'checked_out')
            ->whereDate('check_out_time', today())
            ->sum('total_minutes') / 60;

        // Weekly study data
        $weeklyStudyData = [];
        for ($i = 6; $i >= 0; $i--) {
            $date = now()->subDays($i);
            $hours = SeatBooking::where('user_id', $user->id)
                ->where('status', 'checked_out')
                ->whereDate('check_out_time', $date)
                ->sum('total_minutes') / 60;
            
            $weeklyStudyData[] = [
                'day' => $date->format('D'),
                'hours' => round($hours, 1)
            ];
        }

        // Calculate study streak
        $studyStreak = 0;
        $currentDate = now()->startOfDay();
        while (true) {
            $hasStudy = SeatBooking::where('user_id', $user->id)
                ->where('status', 'checked_out')
                ->whereDate('check_out_time', $currentDate)
                ->exists();
            
            if (!$hasStudy) break;
            $studyStreak++;
            $currentDate = $currentDate->subDay();
        }

        // Session statistics
        $totalSessions = $completedBookings->count();
        $avgSessionDuration = $totalSessions > 0 ? round($totalMinutes / $totalSessions / 60, 1) : 0;
        $focusScore = min(100, round(($totalSessions * 5) + ($studyStreak * 10)));

        return response()->json([
            'stats' => $stats,
            'active_booking' => $activeBooking,
            'recent_activity' => $recentActivity,
            'upcoming_events' => $upcomingEvents,
            'analytics' => [
                'study_streak' => $studyStreak,
                'hours_today' => round($totalHoursToday, 1),
                'weekly_hours' => round($totalMinutes / 60, 1),
                'weekly_study_data' => $weeklyStudyData,
                'avg_session_duration' => $avgSessionDuration,
                'total_sessions' => $totalSessions,
                'focus_score' => $focusScore,
                'weekly_progress' => min(100, round(($totalMinutes / 60) / 20 * 100)), // 20 hours weekly goal
            ]
        ]);
    }
}
